<?php

namespace Modules\Reports\Http\Controllers;

use Exception;
use GuzzleHttp\Client;
use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Modules\Flights\Entities\Flight;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Modules\Hotels\Entities\Hotel;

class ReportsController extends Controller
{
    protected function getClient($isBase = false)
    {
        $headers = [
            'X-Authorization-Token' =>  request()->bearerToken(),
            'X-Access-Token' =>  env('API_TOKEN'),
            'X-Language-Code' => request()->header('X-Language-Code') ?? 'ar',
            'X-Country-Code' => request()->header('X-Country-Code') ?? 'SA',
            'X-Currency-Code' => request()->header('X-Currency-Code') ?? 'SAR',
            'X-Client-Ip' => request()->ip(),
            'Content-Type' => 'application/json',
            'Accept' => 'application/json'
        ];

        if (request()->header('X-Session')) {
            $headers['X-Session'] = request()->header('X-Session');
        }
        $client = new Client([
            'base_uri' => $isBase ? env('API_BASE') : env('API_MODULE_BASE'),
            'headers' => $headers
        ]);

        return $client;
    }
    //======================================Modules & Suppliers ======================
    public function getModules()
    {
        // return env('API_MODULE_BASE');
        $client = $this->getClient();
        try {
            $response = $client->get('users/user_permissions');
        } catch (\Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Internal Server Error'
            ], 500);
        }
        $response = json_decode($response->getBody()->getContents());
        // return $response->data->Permissions;
        if ($response->status) {
            $modules = [];
            foreach ($response->data->Permissions as $module) {
                $moduleData = [
                    'moduleId' => (int) $module->ModuleId,
                    'moduleName' => $module->ModuleName,
                    'suppliers' => []
                ];
                if (!empty($module->Suppliers)) {
                    $suppliers = [];
                    foreach ($module->Suppliers as $supplier) {
                        $suppliers[] = [
                            'supplierId' => (int) $supplier->SupplierId,
                            'supplierName' => $supplier->SupplierName,
                            'moduleName' => $module->ModuleName
                        ];
                    }
                    $moduleData['suppliers'] = $suppliers;
                }
                $modules[] = $moduleData;
            }
        }
        return response()->json([
            'status' => true,
            'data' => [
                'modules' => $modules
            ]
        ]);
    }
    //=== ================================Report Generate ==============================
    public function generate(Request $request)
    {
        $result = $this->getResult($request);
        $totalAmount = $result['hotels']['sum'] + $result['flights']['sum'];
        $mergedList = array_merge(
            empty($result['hotels']['list']) ? [] : $result['hotels']['list']->toArray(),
            $result['flights']['list']
        );
        $result = [
            'list' => $mergedList,
            'total_amount' => $totalAmount,
        ];
        return json_response($result);
    }

    private function getResult(Request $request)
    {
        $module = $request->module;

        if ($module === 'all') {
            $supplier = $request->supplier;
            $isAllSuppliers = $supplier === 'all';

            if (!$isAllSuppliers) {
                $supplierType = explode('_', $supplier)[0];

                if ($supplierType === 'Flights') {
                    return $this->getFlightsResult($request);
                } elseif ($supplierType === 'Hotels') {
                    return $this->getHotelsResult($request);
                }
            }

            $hotels = $this->getHotelsResult($request);
            $flights = $this->getFlightsResult($request);

            return [
                'hotels' => $hotels['hotels'] ?? null,
                'flights' => $flights['flights'] ?? null,
            ];
        }

        $functionName = 'get_' . $module . '_result';
        return $this->{$functionName}($request);
    }

    private function get_user_data($req, $model)
    {
        $user_q = [
            'user' => function ($q) {
                $q->select('id', 'type', 'image', 'first_name', 'last_name', 'company');
            },
            'company' => function ($q) {
                $q->select('id', 'type', 'image', 'company');
            }
        ];

        $user = Auth::user();
        $company_id =  $req->get('parent');
        if ($user->type == 'super_admin') {
            $list = $model::with($user_q);
        } elseif ($user->type == 'company') {
            $list = Flight::with(['user' => $user_q['user']])->where('company_id', $company_id);
        } else {
            $list = Flight::with(['user' => $user_q['user']])->where('user_id', $user->id);
        }
        return $list;
    }

    private function getFlightsResult($req)
    {
        $model = new Flight();
        $list = $this->get_user_data($req, $model);
        $flights = $list
            //when request booking date
            ->when($req->date, function ($query) use ($req) {
                $date = explode('~', $req->date);
                return $query->whereBetween('created_at', [Carbon::parse($date[0])->format('Y-m-d'), Carbon::parse($date[1])->format('Y-m-d')]);
            })
            //when request has supplier
            ->when($req->supplier != 'all' && explode('_', $req->supplier)[0] == 'Flights', function ($query) use ($req) {
                $supplier_name = strtolower(explode('_', $req->supplier)[1]);
                return $query->where('supplier_name', $supplier_name);
            })
            //when req has booking status
            ->when($req->booking_status != 'all', function ($query) use ($req) {
                $booking_status = ucfirst($req->booking_status);
                return $query->where('booking_status', $booking_status);
            })
            //when req has payment status
            ->when($req->payment_status != 'all', function ($query) use ($req) {
                $payment_status = ucfirst($req->payment_status);
                return $query->where('payment_status', $payment_status);
            })
            ->select('user_id', 'company_id', 'supplier_name', 'booking_details', 'booking_status', 'payment_status', 'created_at', 'amount', 'wow_merchant_ref', 'currency_code')
            ->addSelect(DB::raw("'-' as 'hotel'"))
            ->addSelect(DB::raw("'-' as 'hotel_name'"))
            ->addSelect(DB::raw("'-' as 'from_date'"))
            ->addSelect(DB::raw("'-' as 'to_date'"))
            ->addSelect(DB::raw("'flights' as 'booking_type'"))
            ->get();
        return [
            'hotels' => [
                'list' => [],
                'sum' => 0,
            ],
            'flights' => [
                'list' => $this->get_flights_dates($flights),
                'sum' => $flights->sum('amount')
            ],
        ];
    }

    private function get_flights_dates($flights)
    {
        $list = [];
        try {
            $from = [];
            $to = [];
            $list = [];
            foreach ($flights as $flight) {
                $legs = $flight['booking_details']['BookingNeededInfo']['SearchRequest']['Legs'];
                $from = [
                    'code' => $legs[0]['departureCode'],
                    'date' => $legs[0]['outboundDate'],
                ];
                if (count($legs) > 1) {
                    $to = [
                        'code' => $legs[1]['departureCode'],
                        'date' => $legs[1]['outboundDate'],
                    ];
                } else {
                    $to = [
                        'code' => $legs[0]['arrivalCode'],
                        'date' => $legs[0]['outboundDate'],
                    ];
                }
                $flight['from_date'] = $from;
                $flight['to_date'] = $to;
                $list[] = $flight;
            }
        } catch (Exception $e) {
        }
        return $list;
    }

    private function getHotelsResult($req)
    {
        $model = new Hotel();
        $list = $this->get_user_data($req, $model);
        $hotels = $list
            //when request booking date
            ->when($req->date, function ($query) use ($req) {
                $date = explode('~', $req->date);
                return $query->whereBetween('created_at', [Carbon::parse($date[0])->format('Y-m-d'), Carbon::parse($date[1])->format('Y-m-d')]);
            })
            //when request has supplier
            // ->when($req->supplier != 'all', function ($query) use ($req) {
            //     $supplier_name = strtolower($req->supplier);
            //     return $query->where('supplier_name', $supplier_name);
            // })
            //when req has booking status
            ->when($req->booking_status != 'all', function ($query) use ($req) {
                $booking_status = ucfirst($req->booking_status);
                return $query->where('booking_status', $booking_status);
            })
            //when req has payment status
            ->when($req->payment_status != 'all', function ($query) use ($req) {
                return $query->where('payment_status', $req->payment_status);
            })
            ->select('supplier_name', 'user_id', 'company_id', 'booking_status', 'from_date', 'to_date', 'hotel_name', 'payment_status', 'created_at', 'amount', 'wow_merchant_ref', 'currency_code')
            ->addSelect(DB::raw("'-' as 'booking_details'"))
            ->addSelect(DB::raw("'hotels' as 'booking_type'"))
            ->get();
        return [
            'flights' => [
                'list' => [],
                'sum' => 0,
            ],
            'hotels' => [
                'list' => $hotels,
                'sum' => $hotels->sum('amount'),
            ]
        ];
    }
}
