<?php

namespace App\Models;

use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Laravel\Fortify\TwoFactorAuthenticatable;
use Spatie\Permission\Traits\HasRoles;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable, TwoFactorAuthenticatable, HasRoles;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'first_name',
        'last_name',
        'email',
        'type',
        'profile',
        'agent_token',
        'password',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
        'two_factor_recovery_codes',
        'two_factor_secret',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        // 'agent_token' => 'encrypted',
    ];

    public function getFullNameAttribute($value)
    {
        return "{$this->first_name} {$this->last_name}";
    }



    public function balance()
    {
        return $this->hasOne(UserCredit::class, 'user_id');
    }

    public function suppliers()
    {
        return $this->hasMany(UserSupplier::class, 'user_id');
    }

    /*
    * Params-> model_class, table_name, foreign_key, local_key 
    */
    public static function get_roles($user_id)
    {
        $roles =  DB::table('model_has_roles')->select('role_id')->where('model_id', "=", $user_id)->get()->toArray();

        return array_column($roles, 'role_id');
    }
    public function roles()
    {
        return $this->belongsToMany(Role::class, 'model_has_roles', 'model_id', 'role_id');
    }

    public static function get_modules($user_id)
    {
        $modules =  DB::table('user_supplier')->select('module_id')->where('user_id', "=", $user_id)->get()->toArray();

        return array_column($modules, 'module_id');
    }

    public static function get_suppliers($user_id)
    {
        $suppliers =  DB::table('user_supplier')->select('supplier_id')->where('user_id', "=", $user_id)->get()->toArray();
        return array_column($suppliers, 'supplier_id');
    }


    public function scopeAgents($query)
    {
        $agentRole = Role::where('name', 'Agent')->first();

        if ($agentRole) {
            return $query->whereHas('roles', function ($query) use ($agentRole) {
                $query->where('id', $agentRole->id);
            });
        }

        return $query;
    }

    public function isAgent()
    {
        $role = Role::where('name', 'Agent')->first();

        if (!$role) {
            return false;
        }

        return $this->roles()->where('id', $role->id)->where('model_id', $this->id)->exists();
    }


    public function getCreatedAtAttribute($value)
    {
        return (new Carbon($value))->format(config('Reading.date_time_format'));
    }

    public function setCreatedAtAttribute($value)
    {
        $this->attributes['created_at'] = (new Carbon($value))->format('Y-m-d H:i:s');
    }
    public function scopeMarkup()
    {
        $markup = Markup::where('agent_id', $this->id)->where('type', Markup::Type_Agent)->first();
        return $markup->data ?? [];
    }

    public function scopeModules()
    {
        $usersSuppliers = UserSupplier::where('user_id', $this->id)->with('module')->get();

        $moduleKeys = $usersSuppliers->map(function ($userSupplier) {
            return $userSupplier->module->key;
        })->toArray();
        return $moduleKeys;
    }


    public function scopeModulesAndSuppliers()
    {
        $usersSuppliers = UserSupplier::where('user_id', $this->id)
            ->with('module')
            ->with('supplier')
            ->get();

        $moduleSupplierData = $usersSuppliers->groupBy('module_id')->map(function ($moduleSuppliers, $moduleId) {
            $module = $moduleSuppliers[0]->module;

            $suppliers = $moduleSuppliers->map(function ($userSupplier) {
                $supplier = $userSupplier->supplier;
                return [
                    'supplierId' => $supplier->id,
                    'supplierName' => $supplier->name,
                    'supplierKey' => $supplier->key,
                    'supplierSettings' => $userSupplier->supplier_settings,
                    // Add other supplier-related data if needed
                ];
            })->values()->all();

            return [
                'moduleId' => $module->id,
                'moduleName' => $module->name,
                'moduleKey' => $module->key,
                'suppliers' => $suppliers,
            ];
        })->values()->all();

        return $moduleSupplierData;
    }
}
