<?php

namespace Modules\Users\Http\Controllers;

use App\Http\Controllers\Controller;
// use Laravel\Lumen\Routing\Controller as BaseController;
use GuzzleHttp\Client;
use GuzzleHttp\Psr7;
use GuzzleHttp\Exception\RequestException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Modules\Users\Entities\User;
use Modules\Auth\Entities\Userverify;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\URL;
use Spatie\UrlSigner\MD5UrlSigner;
use Exception;
use Illuminate\Support\Str;
use Illuminate\Support\Carbon;
use Illuminate\Support\Env;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Route;
use Modules\Payments\Entities\Payment;
use Modules\Users\Entities\Role;
use Modules\Users\Entities\Permission;

class UsersController extends Controller
{

    public function check_email(Request $req)
    {
        $user = User::where('email', $req->input('email'))->get();
        return json_response([], 200, (bool) count($user), count($user) ? 'Email exists' : 'Email not exists');
    }

    //Company Listing
    public function list_companies(Request $request)
    {
        $parent = $request->get('parent');
        $users = User::where(['parent_id' => $parent, 'type' => 'company'])->get();
        $usersArray = $users->toArray();
        foreach ($users as $key => $user) {
            $usersArray[$key]['groups'] = $user->roles->map->only(['id', 'name', 'desc']);
            $usersArray[$key]['permissions'] =  $user->permissions->map->only(['id', 'name', 'type']);
        }
        return json_response($usersArray, 200, true, 'Success');
    }

    //Employees Listing
    public function list_employees(Request $request)
    {
        $user_id = $request->user()->id;
        $parent_id = $request->get('parent');
        $users = User::where(['parent_id' => $parent_id, 'type' => 'employee'])->where('id', '!=', $user_id)->get();
        $usersArray = $users->toArray();
        foreach ($users as $key => $user) {
            $usersArray[$key]['groups'] = $user->roles->map->only(['id', 'name', 'desc']);
            // $usersArray[$key]['permissions'] =  $user->permissions->map->only(['id', 'name']);
        }
        return json_response($usersArray, 200, true, 'Success');
    }
    //Company Employees Listing
    public function list_company_employees($company_id)
    {
        $users = User::where(['parent_id' => $company_id, 'type' => 'employee'])->get();
        $usersArray = $users->toArray();
        foreach ($users as $key => $user) {
            $usersArray[$key]['groups'] = $user->roles->map->only(['id', 'name', 'desc']);
            // $usersArray[$key]['permissions'] =  $user->permissions->map->only(['id', 'name']);
        }
        return json_response($usersArray, 200, true, 'Success');
    }

    //list B2B companies and their employees
    public function list_companies_with_employees(Request $request)
    {
        $parent = $request->get('parent');
        $users = User::where(['parent_id' => $parent, 'type' => 'company'])->get();
        $usersArray = $users->toArray();
        foreach ($users as $key => $user) {
            $usersArray[$key]['users'] = User::where('parent_id', $user->id)->get();
            $usersArray[$key]['groups'] = $user->roles->map->only(['id', 'name', 'desc']);
            $usersArray[$key]['permissions'] =  $user->permissions->map->only(['id', 'name']);
        }
        return json_response($usersArray, 200, true, 'Success');
    }

    //add company by b2b
    public function add_company(Request $request)
    {
        $parent = $request->get('parent');
        $validatedData = $request->validate([
            'company' => 'required|unique:users,company',
            'mobile' => 'required|min:9|max:15|unique:users,mobile',
            'email' => 'required|unique:users,email',
            'credit_amount' => 'nullable',
            'credit_allowed_amount' => 'nullable',
            'password' => 'required|confirmed',
        ]);
        $validatedData['active'] = $request->active;
        $validatedData['parent_id'] = $parent;
        $validatedData['type'] = 'company';
        $validatedData['title'] = '';
        $validatedData['first_name'] = '';
        $validatedData['last_name'] = '';
        $validatedData['credit_amount'] = $request->credit_amount;
        $validatedData['credit_allowed_amount'] = $request->credit_allowed_amount;
        $validatedData['password'] = Hash::make($validatedData['password']);
        // return $validatedData;
        $company = User::create($validatedData);
        if ($company) {
            $groups = [];
            //create accounting and operations groups
            $groups[] = Role::create(['name' => 'accounting', 'user_id' => $company->id, 'guard_name' => 'sanctum', 'desc' => 'this accounting group for company ' . $company->company]);
            $groups[] = Role::create(['name' => 'operations', 'user_id' => $company->id, 'guard_name' => 'sanctum', 'desc' => 'this operations group for company ' . $company->company]);
            // $company->syncRoles($groups);
            return json_response($company, 200, true, trans('web.added_successfully'));
        }
    }

    //update company by b2b
    public function update_company(Request $request, $id)
    {
        $company = User::where('id', $id)->first();
        $parent = $request->get('parent');
        if ($parent == $company->parent_id) {
            $validatedData = $request->validate([
                'company' => 'required|unique:users,company,' . $id . 'id',
                'mobile' => 'required|min:9|max:15|unique:users,mobile,' . $id . 'id',
                'email' => 'required|unique:users,email,' . $id . 'id',
                'credit_amount' => 'nullable',
                'credit_allowed_amount' => 'nullable',
                'password' => 'confirmed',
            ]);
            $validatedData['active'] = $request->active;
            $validatedData['parent_id'] = $parent;
            $validatedData['title'] = '';
            $validatedData['type'] = 'company';
            $validatedData['first_name'] = '';
            $validatedData['last_name'] = '';
            $validatedData['credit_amount'] = $request->credit_amount;
            $validatedData['credit_allowed_amount'] = $request->credit_allowed_amount;
            if ($request['password']) {
                $validatedData['password'] = Hash::make($validatedData['password']);
            } else {
                $validatedData['password'] = $company->password;
            }
            // return $validatedData;
            $update = $company->update($validatedData);
            if ($update) {
                $roles = $request->validate([
                    'roles' => 'array'
                ]);

                if ($roles) {
                    $company->syncRoles($roles);
                }
                return json_response($company, 200, true, trans('web.updated_successfully'));
            }
        } else {
            return json_response([], 200, false, trans('web.unauthorized'));
        }
    }

    public function add_employee(Request $request, $company_id = null)
    {
        if (!$company_id) {
            $parent = $request->get('parent');
        } else {
            $parent = $company_id;
        }
        $type = 'employee';
        $parent_credit_amount = User::where('id', $parent)->first()->credit_amount;

        $validatedData = $request->validate([
            'first_name' => 'required',
            'last_name' => 'required',
            'mobile' => 'required|min:9|max:15|unique:users,mobile',
            'email' => 'required|unique:users,email',
            'password' => 'required|confirmed',
            'credit_amount' => 'nullable',
            'credit_allowed_amount' => 'nullable',
        ]);

        if ($request->credit_amount > $parent_credit_amount) {
            return json_response([], 200, false, trans('web.credit_error'));
        } else {
            $validatedData['active'] = $request->active;
            $validatedData['parent_id'] = $parent;
            $validatedData['title'] = $request->title;
            $validatedData['type'] = $type;
            $validatedData['credit_amount'] = $request->credit_amount;
            $validatedData['credit_allowed_amount'] = $request->credit_allowed_amount;
            $validatedData['password'] = Hash::make($validatedData['password']);
            $user = User::create($validatedData);
            if ($user) {
                $roles = $request->validate([
                    'roles' => 'array'
                ]);

                if ($roles) {
                    $user->syncRoles($roles);
                }
                $permissions = $request->validate([
                    'permissions' => 'array'
                ]);

                if ($permissions) {
                    $user->syncPermissions($permissions);
                }
                return json_response($user, 200, true, trans('web.added_successfully'));
            }
        }
    }


    public function update_employee(Request $request, $id)
    {
        $admin = Auth::user();
        $employee = User::findOrFail($id);
        $parent = User::where('id', $employee->parent_id)->first();
        $type = 'employee';
        $parent_credit_amount = $parent->credit_amount;
        if ($parent->id == $employee->parent_id || $admin->type == 'super_admin') {
            $validatedData = $request->validate([
                'first_name' => 'required',
                'last_name' => 'required',
                'mobile' => 'required|min:9|max:15|unique:users,mobile,' . $id . 'id',
                'email' => 'required|unique:users,email,' . $id . 'id',
                'password' => 'confirmed',
                'credit_amount' => 'nullable',
                'credit_allowed_amount' => 'nullable',
            ]);
            if ($request->credit_amount > $parent_credit_amount) {
                return json_response([], 200, false, trans('web.credit_error'));
            } else {
                $validatedData['active'] = $request->active;
                $validatedData['parent_id'] = $parent->id;
                $validatedData['title'] = $request->title;
                $validatedData['type'] = $type;
                $validatedData['credit_amount'] = $request->credit_amount;
                $validatedData['credit_allowed_amount'] = $request->credit_allowed_amount;
                if ($request['password']) {
                    $validatedData['password'] = Hash::make($validatedData['password']);
                } else {
                    $validatedData['password'] = $employee->password;
                }
                $user = $employee->update($validatedData);
                if ($user) {
                    $roles = $request->validate([
                        'roles' => 'array'
                    ]);

                    if ($roles) {
                        $employee->syncRoles($roles);
                        $current_roles = $employee->roles;
                        if ($current_roles) {
                            foreach ($current_roles as $role) {
                                $employee->revokePermissionTo($role->permissions);
                            }
                        }
                    }

                    $permissions = $request->validate([
                        'permissions' => 'array'
                    ]);

                    if ($permissions) {
                        $employee->syncPermissions($permissions);
                    }
                    return json_response($employee, 200, true, trans('web.updated_successfully'));
                }
            }
        } else {
            return json_response([], 200, false, trans('web.unauthorized'));
        }
    }

    public function update_credit_amount(Request $request, int $id)
    {
        $request->validate([
            'credit_amount' => 'required',
        ]);
    
        $admin = Auth::user();
        $user = User::findOrFail($id);
        $parent = User::findOrFail($user->parent_id);
        $userType = $user->type;
    
        if ($admin->type !== 'super_admin' && $user->parent_id !== $parent->id) {
            return json_response([], 200, false, trans('web.unauthorized'));
        }
    
        $parentCreditAmount = $parent->credit_amount;
        $userCreditAmount = $user->credit_amount;
        $creditAmount = (int) $request->credit_amount;
        $creditAllowedAmount = (int) $request->credit_allowed_amount ?? 0;
    
        if ($userType === 'company') {
            $user->update([
                'credit_amount' => $userCreditAmount + $creditAmount,
                'credit_allowed_amount' => $creditAllowedAmount,
            ]);
    
            Payment::create([
                'added_by' => $request->user()->id,
                'company_id' => $parent->id,
                'user_id' => $user->id,
                'amount' => $creditAmount,
                'type' => 'credit',
            ]);
    
            return json_response([], 200, true, trans('web.updated_successfully'));
        }
    
        if ($userType === 'employee') {
            if ($creditAmount > $parentCreditAmount) {
                return json_response([], 200, false, trans('web.credit_error'));
            }
    
            $user->update([
                'credit_amount' => $userCreditAmount + $creditAmount,
                'credit_allowed_amount' => $creditAllowedAmount,
            ]);
    
            Payment::create([
                'added_by' => $request->user()->id,
                'user_id' => $user->id,
                'company_id' => $parent->id,
                'amount' => $creditAmount,
                'type' => 'credit',
            ]);
    
            return json_response([], 200, true, trans('web.updated_successfully'));
        }
    }

    public function user_groups($id, Request $request)
    {
        $parent_id = $request->get('parent');
        $admin = Auth::user();
        $user = User::findOrFail($id);
        if ($user->parent_id == $parent_id || $admin->type = "super_admin") {
            $roles = [];
            $user_roles = $user->roles;
            $user->getPermissionsViaRoles();
            foreach ($user_roles as $role) {
                $roles[$role->id] = [
                    'id' => $role->id,
                    'name' => $role->name,
                    'permissions' => [],
                    'actions' => []
                ];

                foreach ($role->permissions as $row) {
                    if (strpos($row['name'], '.') !== false) {
                        $roles[$role->id]['permissions'][] = $row['name'];
                    } else {
                        $roles[$role->id]['actions'][] = $row['name'];
                    }
                }
            }
            return json_response(array_values($roles));
        } else {
            return json_response([], 200, false, trans('web.unauthorized'));
        }
    }

    public function activate(Request $request, $id)
    {
        $parent = $request->get('parent');
        //check if logged in user own this user or company
        $user = User::findOrFail($id);
        if ($user->parent_id == $parent) {
            $check = User::where('id', $id)->update(['active' => true]);
            return response()->json([
                'status'    =>  $check ? true : false,
                'message' => 'activated successfully',
            ]);
        } else {
            return json_response([], 200, false, trans('web.unauthorized'));
        }
    }

    public function deactivate(Request $request, $id)
    {
        $parent = $request->get('parent');
        //check if logged in user own this user or company
        $user = User::findOrFail($id);
        if ($user->parent_id == $parent) {
            $check = User::where('id', $id)->update(['active' => false]);
            return response()->json([
                'status'    =>  $check ? true : false,
                'message' => 'deactivated successfully',
            ]);
        } else {
            return json_response([], 200, false, trans('web.unauthorized'));
        }
    }

    public function delete($id, Request $request)
    {
        $parent = $request->get('parent');
        $entity = User::findOrFail($id);
        $admin = Auth::user();
        if ($entity->parent_id == $parent ||  $admin->type == 'super_admin') {
            //delete company and company's employees
            if ($entity->type == 'company') {
                $company_users = User::where('parent_id', $entity->id)->get();
                foreach ($company_users as $user) {
                    User::findOrFail($user->id)->delete();
                }
                $entity->delete();
                return json_response([], 200, true, trans('web.deleted_successfully'));
            }
            if ($entity->type == "employee") {
                $entity->delete();
                return json_response([], 200, true, trans('web.deleted_successfully'));
            }
        } else {
            return json_response([], 200, false, trans('web.unauthorized'));
        }
    }
}
