<?php

namespace Modules\Auth\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;
use Illuminate\Auth\Events\PasswordReset;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\Auth;
use Illuminate\Mail\Message;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Carbon;
use Modules\Auth\Entities\Userverify;
use Illuminate\Support\Facades\Mail;
use Spatie\UrlSigner\MD5UrlSigner;
use Exception;
use Modules\Users\Entities\User;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Http;
use Modules\Users\Entities\User as EntitiesUser;
use GuzzleHttp\Client;
use Modules\Users\Http\Controllers\PermissionsController;
use Intervention\Image\Facades\Image as Image;
use Illuminate\Support\Facades\Storage;
use Modules\Common\Http\Controllers\NotificationsController;
use Modules\Locales\Entities\Currency;
use Modules\Settings\Entities\Settings;

class AuthController extends Controller
{
    public $client;
    public function __construct(Request $req)
    {
        $headers = [
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . env('API_TOKEN'), // Assuming env() function retrieves the API_TOKEN from environment variables
            'X-Language-Code' => $req->header('X-Language-Code'),
            'X-Country-Code' => $req->header('X-Country-Code'),
            'X-Currency-Code' => $req->header('X-Currency-Code'),
            'X-Client-IP' => $req->ip()
        ];

        if ($req->header('X-Session')) {
            $headers['X-Session'] = $req->header('X-Session');
        }

        $this->client = new Client([
            'base_uri' => env('API_BASE'),
            'headers' => $headers
        ]);
    }

    //register
    public function register(Request $request)
    {
        $validatedData = $request->validate([
            'first_name' => 'required|alpha|max:20',
            'last_name' => 'required|alpha|max:20',
            'email' => 'required|string|email|max:100|unique:users',
            'mobile' => 'required|unique:users',
            'password' => 'required',
        ]);

        $user = User::create([
            'first_name' => $validatedData['first_name'],
            'last_name' => $validatedData['last_name'],
            'email' => $validatedData['email'],
            'mobile' => $validatedData['mobile'],
            'password' => Hash::make($validatedData['password']),
        ]);

        // event(new Registered($user));
        $token = $user->createToken('auth_token')->plainTextToken;

        $verify_token = Str::random(64);
        UserVerify::create([
            'user_id' => $user->id,
            'token' => $verify_token
        ]);

        //create signed url for email activation
        $urlSigner = new MD5UrlSigner(env('APP_KEY'));
        $url = $urlSigner->sign(env('FRONT_URL') . '/auth/activate/' . $user->id . '/' . $verify_token, Carbon::now()->addHours(1));
        try {
            $mail = Mail::send('emails.emailVerificationEmail', ['url' => $url], function ($message) use ($request) {
                $message->to($request->email);
                $message->subject('Email Verification Mail');
            });
        } catch (Exception $e) {
            return response()->json([
                'status' => 'success',
                'message' => 'user registered  but can\'t send verification email',
            ], 500);
        }
        return response()->json([
            'access_token' => $token,
            'mail' => $mail,
            'data' => $token,
            'status' => 'success',
        ], 200);
    }
    //update user
    public function update(Request $request)
    {
        $user = $request->user();
        $type = $user->type;
        if ($type == 'super_admin') {
        } else {
            $user = $request->user();
            $validatedData = $request->validate([
                'first_name' => 'required',
                'last_name' => 'required',
                'email' => 'required|unique:users,email,' . $user->id . ',id',
                'mobile' => 'required|unique:users,mobile,' . $user->id . ',id',
            ]);

            if ($user) {
                $user->first_name = $validatedData['first_name'];
                $user->last_name = $validatedData['last_name'];
                // $user->email = $validatedData['email'];
                $user->mobile = $validatedData['mobile'];
                if ($user->update()) {
                    return json_response(null, 200, true, trans('web.updated_successfully'));
                }
            }
            return response()->json([
                'status' => 'error',
                'message' => 'user not found'
            ], 401);
        }
    }

    //update password
    public function updatePassword(Request $request)
    {
        $user = $request->user();
        $type = $user->type;

        $validated = $request->validate(
            [
                'current_password' => 'required',
                'new_password' => 'required|different:current_password|required_with:new_password_confirmation|same:new_password_confirmation',
            ],
            [
                // "current_password.required" => 'كلمة المرور الحالية مطلوبة',
                // "new_password.required" => 'كلمة المرور الجديدة مطلوبة',
                // "new_password.same" => 'تأكيد كلمة المرور غير متشابهه',
                // "new_password.different" => 'كلمة المرور الحالية والجديدة متشابهين',
            ]
        );

        if ($type == 'super_admin') {
            $response = $this->client->post('users/change_password', ['json' => $request->all()]);
            return response((string) $response->getBody(), $response->getStatusCode());
        } else {

            //check current password is correct
            if (!Hash::check($request->current_password, $user->password)) {
                return response()->json([
                    'status' => 'failed',
                    'message' => 'current password is wrong',
                ], 402);
            }

            if (Hash::check($request->new_password, $user->password)) {
                return response()->json([
                    'status' => 'failed',
                    'message' => 'can\'t use the same password',
                ], 402);
            } else {
                $user->update([
                    'password' => Hash::make($request->new_password),
                ]);
                return json_response(null, 200, true, trans('web.updated_successfully'));
            }
        }
    }
    //update Image
    public function updateImage(Request $request)
    {
        $user = $request->user();

        $request->validate([
            'image' => 'image|mimes:jpeg,png,jpg,gif,svg|max:2048|nullable',
        ]);
        $request_data = $request->except(['image']);
        $user_path = Storage::disk('user')->path('/');
        //icon
        if ($request->hasfile('image')) {
            $name = $request->file('image')->hashName();
            $request_data['image'] = $name;
            Image::make($request->file('image')->getRealPath())->save($user_path . $name)->response();
        }
        $user->update($request_data);
        return json_response(null, 200, true, trans('web.updated_successfully'));
    }

    //login
    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required',
            'password' => 'required',
        ]);
        // $notifications = new NotificationsController();
        // Check if user is the admin
        if ($request->email == env('ADMIN_EMAIL')) {
            $response = $this->client->post('login', ['json' => $request->all()]);
            $response = json_decode($response->getBody()->getContents());
            if (!$response->status && !$response->data) {
                return response()->json([
                    'message' => $response->message,
                    'status' => false,
                ], 200);
            }

            // Get b2b profile and store in users
            // $profileResponse = Http::withHeaders([
            //     'X-Access-Token' => env('API_TOKEN'),
            // ])->get(env('API_MODULE_BASE') . '/users/profile');
            // $profileResponse = json_decode($profileResponse->getBody()->getContents());
            $data = $response->data;
            if ($data->user->email != env('ADMIN_EMAIL')) {
                return response()->json([
                    'message' => trans('web.wrong_login_configuration'),
                    'status' => false,
                ], 200);
            }
            // return $data->user;
            // Store b2b as user with type admin in local DB
            $user = EntitiesUser::updateOrCreate([
                'email' => env('ADMIN_EMAIL'),
            ], [
                'id' => 1,
                'first_name' => $data->user->first_name,
                'last_name' => $data->user->last_name,
                // 'company' => $data->user->company,
                'email' => $data->user->email,
                // 'mobile' => $data->user->phone,
                'type' => 'super_admin',
                'credit_amount' => optional($data->user->balance)->credit_amount,
                'credit_allowed_amount' => optional($data->user->balance)->credit_allowed_amount,
                'modules' => $data->user->modules,
                'password' => '',
            ]);

            // if ($user->status) {
            //     return json_response(null, 200, false, trans('web.user_not_active'));
            // }

            // Check if the user is already logged in
            $existingToken = $user->tokens()->where('name', 'auth_token')->first();
            if ($existingToken) {
                $existingToken->delete();
            }
            $token = $user->createToken('auth_token')->plainTextToken;
            return json_response($token, 200, true, trans('web.logged_successfully'));
        }

        // Ordinary user
        if (!Auth::attempt($request->only('email', 'password'))) {
            return response()->json([
                'message' => trans('web.invalid_login_details'),
            ], 401);
        }

        $user = User::where('email', $request->email)->firstOrFail();
        if ($user->active == 0) {
            return json_response(null, 200, false, trans('web.user_not_active'));
        }
        $existingToken = $user->tokens()->where('name', 'auth_token')->first();
        if ($existingToken) {
            $existingToken->delete();
        }
        $token = $user->createToken('auth_token')->plainTextToken;
        return json_response($token, 200, true, trans('web.logged_successfully'));
    }

    //logout
    public function logout(Request $request)
    {
        // Revoke the token that was used to authenticate the current request
        $request->user()->currentAccessToken()->delete();
        //$request->user->tokens()->delete(); // use this to revoke all tokens (logout from all devices)
        return response()->json(null, 200);
    }

    /*
	 * Get authenticated user details
	*/
    public function getAuthenticatedUser(Request $request)
    {
        // Sync permissions
        // $permission_controller = new PermissionsController($request);
        // $permission_controller->SyncPermissionsFromBackOffice();
        $user = auth('sanctum')->user();

        // Check if user is an admin
        if ($user->email == env('ADMIN_EMAIL')) {
            try {
                $response = Http::withHeaders([
                    'Authorization' => 'Bearer ' . env('API_TOKEN'),
                ])->get(env('API_BASE') . 'users/profile');
                $response = json_decode($response->getBody()->getContents());
                $status = (bool) $response->status;
                $message = $response->message;
                $data = $response->data;
            } catch (\Exception $e) {
                return json_response([], 500, false, $e->getMessage());
            }

            // Get user permissions
            $permissions = [];
            foreach ($user->getAllPermissions() as $row) {
                if (strpos($row['name'], '_') !== false) {
                    $module = explode('_', $row['name'])[0];
                    $modules[$module] = $module;
                }
                $permissions[] = $row['name'];
            }

            $currency = $data->currency;
            $user->update([
                'credit_amount' => updateCurrencyRate($data->credit_amount, $currency),
                'credit_allowed_amount' => updateCurrencyRate($data->credit_allowed_amount, $currency),
            ]);

            // Append modules, permissions, and roles to user data
            $user->modules = (object) array_merge((array) $data->modules, ['markups' => 'markups']);
            $user = $user->toArray();
            $user['permissions'] = $permissions;
            $user['roles'] = 'super_admin';
            $user['base_currency'] = getBaseCurrency();

            // Return user data with status and message
            return json_response($user, 200, $status, $message);
        } else {
            // Get ordinary user profile
            $user->settings = json_decode($user->settings);
            $modules = [];
            $permissions = [];
            foreach ($user->getAllPermissions() as $row) {
                if (strpos($row['name'], '_') !== false) {
                    $module = explode('_', $row['name'])[0];
                    $modules[$module] = $module;
                }
                $permissions[] = $row['name'];
            }

            // Update user data with modules and permissions
            $user->modules = $modules;
            $user = $user->toArray();
            $user['permissions'] = $permissions;

            // Return user data
            if ($user) {
                return json_response($user);
            } else {
                return json_response([], 401, false, 'Invalid Token');
            }
        }
    }

    public function sendPasswordResetLinkEmail(Request $request)
    {
        $request->validate(['email' => 'required|email']);

        $status = Password::sendResetLink(
            $request->only('email')
        );

        if ($status === Password::RESET_LINK_SENT) {
            return response()->json(['message' => __($status), 'status' => 'email sent'], 200);
        } else {
            // throw ValidationException::withMessages([
            // 	'email' => __($status)
            // ]);
            return response()->json(['message' => 'Can\'t send Email'], 503);
        }
    }

    public function validateToken(Request $request)
    {
        $reset = DB::table("password_resets")->where('email', $request->email)->first();
        if ($reset) {
            if ($this->hasher->check($request->token, $reset->token)) {
                $user = User::where('email', $request->email)->first();
                return response()->json([
                    'data' => $user,
                    'status' => 'success',
                    'message' => 'Valid Token!',
                    // 'token_type' => 'Bearer',
                ], 200);
            } else {
                return response()->json([
                    'message' => "Invalid token",
                ], 401);
            }
        } else {
            return response()->json([
                'message' => "Invalid Token",
            ], 401);
        }
    }

    public function resetPassword(Request $request)
    {
        $status = Password::reset(
            $request->only('email', 'password', 'token'),
            function ($user, $password) use ($request) {
                $user->forceFill([
                    'password' => Hash::make($password)
                ])->setRememberToken(Str::random(60));

                $user->save();

                event(new PasswordReset($user));
            }
        );

        if ($status == Password::PASSWORD_RESET) {
            return response()->json(['message' => __($status)], 200);
        } else {
            throw ValidationException::withMessages([
                'email' => __($status)
            ]);
        }
    }
    //check siged url
    public function checksignedToken(Request $request)
    {
        $urlSigner = new MD5UrlSigner(env('APP_KEY'));
        if ($urlSigner->validate($request->url)) {
            return response()->json([
                'status' => 'success',
                'message' => 'Valid Signature',
            ], 200);
        } else {
            return response()->json([
                'message' => 'Not Valid Signature',
            ], 200);
        }
    }

    public function verifyAccount($id, $token)
    {
        $verifyUser = UserVerify::where('token', $token)->first();
        if (!is_null($verifyUser)) {
            $user = $verifyUser->user;
            if (!$user->is_email_verified) {
                $verifyUser->user->is_email_verified = 1;
                $verifyUser->user->save();
                // UserVerify::destroy ($verifyUser->id);
                return response()->json([
                    'status' => 'success',
                    'message' => 'Email Varified!',
                ], 200);
            } else {
                return response()->json([
                    'status' => 'fail',
                    'message' => 'your email already verified',
                ]);
            }
        } else {
            return response()->json([
                'status' => 'fail',
                'message' => 'Invalid Token',
            ]);
        }
    }

    public function updateAppSettings(Request $request)
    {
        $user = Auth::user();
        if ($user->email == env('ADMIN_EMAIL')) {
            $validated = $request->validate([
                'frontend_url' => 'required|url',
                'backend_url' => 'required|url',
                'access_token' => 'required|uuid',
                'password' => 'required'
            ]);
            $response = $this->client->post('users/settings_edit', ['json' => $validated, 'http_errors' => false]);
            return response((string) $response->getBody(), $response->getStatusCode());
        }
        return json_response('', 401, false, 'Authentication failed');
    }

    public function getAppSettings()
    {
        $user = Auth::user();
        if ($user->email == env('ADMIN_EMAIL')) {
            $response = $this->client->get('users/app_settings');
            return response((string) $response->getBody(), $response->getStatusCode());
        }
        return json_response('', 401, false, 'Authentication failed');
    }

    public function get_user_modules_with_suppliers()
    {
        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . env('API_TOKEN'),
        ])->get(env('API_BASE') . 'users/modules');
        return $response;
    }
}
