<?php

namespace Modules\Hotels\Http\Controllers;

use App\Models\CityCode;
use App\Models\Customer;
use App\Models\Supplier;
use App\Models\User;
use App\Models\WowSession;
use Exception;
use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Modules\StaticData\Entities\Country;
use GuzzleHttp\Client;
use GuzzleHttp\Handler\CurlMultiHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Promise\Utils;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Validator;
use Modules\Hotels\Entities\HotelsBookingList;
use Modules\Hotels\Entities\HotelSearch;
use Modules\StaticData\Entities\City;
use Modules\StaticData\Entities\CityTranslation;
use Modules\StaticData\Entities\Hotel;
use Modules\StaticData\Entities\HotelTranslation;
use Modules\StaticData\Entities\Nationality;
use Modules\StaticData\Entities\SupplierCity;
use Modules\StaticData\Entities\SupplierHotel;
use Psr\Http\Message\RequestInterface;
use Symfony\Component\Process\ExecutableFinder;
use Symfony\Component\Process\Process;

class HotelsController extends Controller
{
    protected $suppliers;
    protected $headers;
    protected $moduleKey = 'hotels';
    protected $lang;
    protected $currency;
    protected $country;
    protected $resultLimit;
    protected $headerSession;
    protected $ipAddress;

    public function __construct()
    {
        $headers = request()->header();
        $this->lang = $headers['x-language-code'][0] ?? null;
        $this->country = $headers['x-country-code'][0] ?? null;
        $this->resultLimit = $headers['result-limit'][0] ?? 0;
        $this->currency = $headers['x-currency-code'][0] ?? null;
        $this->headerSession = $headers['x-session'][0] ?? null;
        $this->ipAddress = $headers['x-client-ip'][0] ?? null;
    }

    //=================================== AutoComplete =====================================

    public function auto_complete(Request $request)
    {
        $q = urldecode($request->q);
        if (empty($q)) {
            return json_response(['success' => false, 'message' => "Empty Search Query"], 200);
        }

        $cities = $this->searchCities($q)->toArray(); // Convert to array
        $hotels = $this->searchHotels($q)->toArray(); // Convert to array

        // Merge the arrays
        $response = array_merge($cities, $hotels);

        return [
            "status" => true,
            "message" => "Success",
            "data" => $response
        ];
    }

    private function searchCities($q)
    {
        $cityQuery = CityTranslation::with('city')
            ->where('name', 'like', '%' . $q . '%')
            ->limit(10)
            ->get();

        return $cityQuery->map(function ($row) {
            try {
                return [
                    "Type" => "city",
                    "Id" => $row->city->id,
                    "Name" => $row->city->name,
                    "CountryCode" => $row->city->country_code ?? '',
                    "CountryName" => optional($row->city->country)->name ?? '',
                    "UrlCityName" => optional($row->city->translate('en'))->name ?? '',
                    "UrlCountryName" => optional($row->city->city_country->translate('en'))->name ?? ''
                ];
            } catch (Exception $e) {
            }
        });
    }

    private function searchHotels($q)
    {
        $hotelQuery = HotelTranslation::with('hotel')
            ->where('name', 'like', '%' . $q . '%')
            ->limit(10)
            ->get();

        return $hotelQuery->map(function ($row) {
            try {
                return [
                    "Type" => "hotel",
                    "Id" => $row->hotel->id,
                    "Name" => $row->name,
                    "CountryCode" => optional($row->hotel->hotel_city->country_code) ?? '',
                    "CountryName" => optional($row->hotel->hotel_city->country)->name ?? '',
                    "UrlCityName" => optional($row->hotel->hotel_city->translate('en'))->name ?? '',
                    "UrlCountryName" => optional($row->hotel->hotel_city->city_country->translate('en'))->name ?? ''
                ];
            } catch (Exception $e) {
            }
        });
    }

    //============================================= Search ===================================

    public function search(Request $request)
    {
        // try {
        $agent = $request->agent;
        $user_suppliers = getSupplierList($this->moduleKey, $agent);
        $this->suppliers = load_suppliers($user_suppliers, $this->moduleKey);
        if (empty($user_suppliers)) {
            throw new Exception('Agent isn\'t subscribed to any of the hotel suppliers');
        }

        if (empty($request->type) || empty($request->id) || !in_array($request->type, ['city', 'hotel'])) {
            throw new Exception('Search type is empty or undefined');
        }

        //prepare input data
        $input_data['params'] = $this->prepare_common_input_data($request);
        $search_request_handlers = [];

        //prepare city and hotel codes from request id & type
        $type = $request->type;
        $id = $request->id;
        $supplier_codes = $this->prepare_city_and_hotel_code($user_suppliers, $type, $id);

        //merge city & hotel codes with params 
        foreach ($supplier_codes as $supplierKey => $supplier_code) {

            $params = [
                'countryName' => $supplier_code['country']['name'],
                'countryCode' => $supplier_code['country']['code'],
                'cityId' => $supplier_code['city']['id'],
                'cityCode' => $supplier_code['city']['code'],
                'cityName' => $supplier_code['city']['name'],
                'hotelCodes' => $supplier_code['hotel']['codes'],
                'supplierKey' => $supplierKey,
            ];
            $input_data['params'] = array_merge($input_data['params'], $params);
            $request->merge($input_data);
            $supplierClassObject = load_supplier($supplierKey, $this->moduleKey);
            $handlers = $supplierClassObject->getCurlFunc($request->params);
            $search_request_handlers[$supplierKey] = $handlers;
        }
        $search_response = $this->runNodeScript($search_request_handlers);
        // return $search_response;
        // $search_response = $this->handle_curl_multi_search($search_request_handlers); //OLD SENARIO
        $search_results = $this->prepare_search_results($search_response, $request->params);
        return json_response([
            'status' => !empty($search_results),
            'message' => !empty($search_results['Hotels']) ? 'Search results success' : 'No results found. Please try a different search',
            'data' => !empty($search_results) ? $search_results : [],
        ], 200);
        /* } catch (Exception $e) {
            return json_response([
                'status' => false,
                'message' => 'Error in hotels search : ' . $e->getMessage(),
            ], 200);
        }*/
    }

    public function prepare_common_input_data($request)
    {
        return [
            'langCode'          =>  $this->lang,
            'resultLimit'       =>  $this->resultLimit,
            'Checkin'           =>  $request->checkin,
            'Checkout'          =>  $request->checkout,
            'Rooms'             =>  $request->rooms,
            'Type'             =>  $request->type,
            // 'guestNationality'  =>   isset($request->nationality) ? $request->nationality : getNationality($this->country), //TODO GET THE TBL
            'guestNationality'  => $request->nationality,
            'currencyCode'      =>  $this->currency,
            'agentId' => $request->agent->id
        ];
    }

    public function prepare_city_and_hotel_code($user_suppliers, $type, $id)
    {
        $supplier_codes = [];
        if ($type == 'city') {
            try {
                $city = City::findOrFail($id);
            } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
                throw new Exception('City ID is wrong.');
            }

            foreach ($user_suppliers as $supplier) {
                $supplierCity = SupplierCity::where('city_id', $id)->where('supplier_id', $supplier['id'])->first();
                if ($supplierCity) {
                    $supplier_codes[$supplier['key']]['country'] = [
                        'name' => optional($city->city_country->translate('en'))->name ?? '',
                        'code' => optional($city->city_country)->code ?? '',
                    ];
                    $supplier_codes[$supplier['key']]['city'] = [
                        'id' => $city->id,
                        'name' => optional($city->translate('en'))->name ?? '',
                        'code' => $supplierCity->code
                    ];
                    $supplier_codes[$supplier['key']]['hotel']['codes'] = $this->getHotelCodes($supplier['id'], $city->id);
                }
            }

            return $supplier_codes;
        } elseif ($type == 'hotel') {
            try {
                $hotel = Hotel::findOrFail($id);
            } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
                throw new Exception('Hotel ID is wrong.');
            }
            foreach ($user_suppliers as $supplier) {
                $supplier_codes[$supplier['key']]['country'] = [
                    'name' => optional($hotel->hotel_city->city_country->translate('en'))->name ?? '',
                    'code' => optional($hotel->hotel_city->city_country)->code ?? '',
                ];
                $supplier_codes[$supplier['key']]['city'] = [
                    'id' => optional($hotel->hotel_city)->id,
                    'name' => optional($hotel->hotel_city->translate('en'))->name,
                    'code' => optional($hotel->hotel_city)->supplier_code[$supplier['id']] ?? ''
                ];
                $supplier_codes[$supplier['key']]['hotel']['codes'] = $this->getHotelCodes($supplier['id'], null, $id);
            }
            return $supplier_codes;
        }
    }

    private function getHotelCodes($supplier_id, $city_id, $hotel_id = null)
    {
        $query = SupplierHotel::where('supplier_id', $supplier_id)
            ->whereHas('hotel', function ($query) use ($city_id) {
                if ($city_id !== null) {
                    $query->where('city_id', $city_id);
                }
            });

        if ($hotel_id !== null) {
            $query->where('hotel_id', $hotel_id);
        }

        $hotelCodes = $query->pluck('code')->toArray();

        return $hotelCodes;
    }

    public function handle_curl_multi_search($req_handles)
    {
        // Create a new multi-curl handle
        $multi_handle = curl_multi_init();
        // Add all the curl handles to the multi-curl handle

        foreach ($req_handles as $ch) {
            curl_multi_add_handle($multi_handle, $ch);
        }

        // Execute the multi-curl requests
        $active = null;
        do {
            $mrc = curl_multi_exec($multi_handle, $active);
        } while ($mrc == CURLM_CALL_MULTI_PERFORM);

        while ($active && $mrc == CURLM_OK) {
            if (curl_multi_select($multi_handle) == -1) {
                usleep(100);
            }

            do {
                $mrc = curl_multi_exec($multi_handle, $active);
            } while ($mrc == CURLM_CALL_MULTI_PERFORM);
        }

        // Iterate through the curl handles and process the responses
        $results = [];
        foreach ($req_handles as $supplierKey => $ch) {
            $results[$supplierKey] = curl_multi_getcontent($ch);
            curl_multi_remove_handle($multi_handle, $ch);
        }

        // Close the multi-curl handle
        curl_multi_close($multi_handle);
        return $results;
    }

    public function prepare_search_results($req_results, $params)
    {
        $file_name = 'hotels_search_results.json';
        $search_results = [
            'Sessions' => [],
            'Hotels' => [],
            'Info' => []
        ];

        $supp_keys = array_keys($req_results);
        foreach ($this->suppliers as $supplierKey => $supplier) {
            if (in_array($supplierKey, $supp_keys)) {
                $response = $req_results[$supplierKey];
                // Prepare search results using the supplier
                $result = $supplier->prepareSearchResults($response, $params);

                if (isset($result['status'])) {
                    // Update search results
                    $search_results['Hotels'] += $result['data'];
                    $search_results['Info'] += $result['info'] ?? [];

                    if (isset($result['session_key'])) {
                        $search_results['Sessions'][$result['supplier_key']] = $result['session_key'];
                    }

                    // Save results to a file
                    //save_to_file($file_name, $this->moduleKey, $supplierKey, $search_results);

                    // Set session data
                    set_session(
                        $this->moduleKey,
                        $supplierKey,
                        $result['session_key'],
                        $params,
                        $result['data']
                    );
                }
            }
        }

        return $search_results;
    }

    public function runNodeScript($req_handles)
    {
        $response = Http::post('http://localhost:3777/make-axios-request', [
            'req_handles' => $req_handles,
        ]);
        return $response->json();
    }

    //========================================== Available Rooms ============================
    public function available_rooms(Request $request)
    {
        try {
            $sessionData = get_hotel_session_data($request);
            $supplierKey = $sessionData['session']->supplier;
            $hotelCode = $sessionData['hotel_id'];
            $supplierClassObject = load_supplier($supplierKey, $this->moduleKey);
            $sessionSearchData['hotel_id'] = $hotelCode;
            $sessionSearchData['session_key'] = $request->session_id;
            $sessionSearchData['result_index'] = $sessionData['result_index'];
            $sessionSearchData['query'] = $sessionData['session']->query;
            $localHotelDetails = $this->get_local_hotel_details($supplierKey, $hotelCode);
            $availableRoomResponse = $supplierClassObject->available_rooms($sessionSearchData);
            $availableRoomResponse['hotelInfo'] = $localHotelDetails;
            $availableRoomResponse['SessionId'] = $request->session_id;
            $availableRoomResponse['CityWowCode'] = $localHotelDetails['HotelCityId'];
            $availableRoomResponse['HotelWowCode'] = $localHotelDetails['HotelId'];
            $availableRoomResponse['SupplierKey'] = $supplierKey;
            return json_response([
                'status' => true,
                'message' => 'Available Rooms Successfully',
                'data' => $availableRoomResponse
            ], 200);
        } catch (Exception $e) {
            return json_response([
                'status' => false,
                'data' => null,
                'message' => 'Error in Available rooms, ' . $e->getMessage()
            ], 200);
        }
    }

    //========================================== Local Hotel Details  =======================
    public function get_local_hotel_details($supplierKey, $hotelCode)
    {
        try {
            $supplierId = get_supplier_id($supplierKey);
            $hotel = Hotel::getHotelsBySupplierCode($supplierId, $hotelCode)->first();
            if (!$hotel) {
                throw new Exception('Hotel data not found!');
            }

            $data = [];
            $hotel_map = [];
            if ($hotel->lat && $hotel->lng) {
                $hotel_map = [$hotel->lat, $hotel->lng];
            }

            $hotel_country_name = $hotel->hotel_city->city_country->name;
            $hotel_city_name = $hotel->hotel_city->name;
            $hotel_name = $hotel->name;
            $hotel_facilities = $hotel->facilities;
            $hotel_room_facilities = $hotel->room_facilities;
            $hotel_attractions = $hotel->attractions;
            $hotel_address = $hotel->address;
            $hotel_services = $hotel->services;
            $hotel_policy = $hotel->policy;
            $hotel_desc = $hotel->description;

            $data = [
                "HotelId" => $hotel->id,
                "HotelCountryId" => $hotel->hotel_city->city_country->id,
                "HotelCityId" => $hotel->hotel_city->id,
                "HotelCountryName" => $hotel_country_name,
                "HotelCityName" => $hotel_city_name,
                "HotelName" => $hotel_name,
                "HotelRating" => $hotel->rating,
                "HotelTripadvisorRating" => $hotel->tripadvisor_rating,
                "HotelTraipadvisorReviewUrl" => $hotel->tripadvisor_review_url,
                "HotelPhone" => $hotel->phone,
                "HotelFax" => $hotel->fax,
                "HotelWebsite" => $hotel->website,
                "HotelEmail" => $hotel->email,
                "HotelPincode" => $hotel->pincode,
                "HotelAddress" => $hotel_address,
                "HotelFacilities" => $hotel_facilities,
                "HotelRoomFacilities" => $hotel_room_facilities,
                "HotelAttractions" => $hotel_attractions,
                "HotelServices" => $hotel_services,
                "HotelPolicy" => $hotel_policy,
                "HotelDesc" => $hotel_desc,
                "HotelMap" => $hotel_map,
                "Files" => $hotel->gallery,
            ];

            return $data;
        } catch (Exception $e) {
            throw new Exception('error loading hotel data' . $e->getMessage());
        }
    }
    //========================================== Pricing ====================================
    public function pricing(Request $request)
    {
        try {
            $sessionData = get_hotel_session_data($request);
            $supplierKey = $sessionData['session']->supplier;
            $supplierClassObject = load_supplier($supplierKey, $this->moduleKey);
            $sessionSearchData = $sessionData['session']->hotels_search;
            $hotelCode = $sessionData['hotel_id'];
            $sessionSearchData['hotel_id'] = $hotelCode;
            $sessionSearchData['session_key'] = $sessionData['session']->key;
            $sessionSearchData['result_index'] = $sessionData['result_index'];
            $sessionSearchData['room_block'] = $request->room_block;
            $sessionSearchData['session_data'] = $sessionData['session']['data'];
            $pricingResponse = $supplierClassObject->pricing($sessionSearchData);
            $localHotelDetails = $this->get_local_hotel_details($supplierKey, $hotelCode);
            $pricingResponse['preBook'] = [
                'HotelName' => $localHotelDetails['HotelName'],
                'HotelCountry' => $localHotelDetails['HotelCountryName'],
                'HotelCity' => $localHotelDetails['HotelCityName'],
                'HotelAddress' => $localHotelDetails['HotelAddress'],
                'HotelRating' => $localHotelDetails['HotelRating'],
                'RoomsGuests' => $sessionData['session']['query']['Rooms'],
            ];
            return json_response([
                'status' => true,
                'message' => 'Pricing Request Successfull',
                'data' => $pricingResponse
            ], 200);
        } catch (Exception $e) {
            return json_response([
                'status' => false,
                'message' => 'Error in Pricing Request , ' . $e->getMessage()
            ], 200);
        }
    }
    //========================================== Pre Booking ====================================
    // public function pre_booking(Request $request)
    // {
    //     try {
    //         $sessionData = get_hotel_session_data($request);
    //         $supplierKey = $sessionData['session']->supplier;
    //         $supplierClassObject = load_supplier($supplierKey, $this->moduleKey);
    //         $sessionSearchData = $sessionData['session']->hotels_search;
    //         $sessionSearchData['hotel_id'] = $sessionData['hotel_id'];
    //         $sessionSearchData['session_key'] = $sessionData['session']->key;
    //         $sessionSearchData['result_index'] = $sessionData['result_index'];
    //         $sessionSearchData['room_block'] = $request->room_block;
    //         $sessionSearchData['session_data'] = $sessionData['session']['data'];
    //         $pricingResponse = $supplierClassObject->pricing($sessionSearchData);
    //         return json_response([
    //             'status' => true,
    //             'message' => 'Pre Booking Request Successfull',
    //             'data' => $pricingResponse
    //         ], 200);
    //     } catch (Exception $e) {
    //         return json_response([
    //             'status' => false,
    //             'message' => 'Error in Pre Booking Request , ' . $e->getMessage()
    //         ], 200);
    //     }

    //     if ($wow_session = $this->api->wow_sessions->get_module_session($this->api->get_session_id())) {
    //         $supplier = $wow_session['supplier'];
    //         $pre_booking = $this->suppliers[$supplier]->pre_booking($input);
    //         if ($pre_booking) {
    //             $data = [
    //                 'status' => TRUE,
    //                 'message' => 'Success',
    //                 'data' => $pre_booking
    //             ];
    //         }
    //     }

    //     $this->api->response($data);
    // }
    //========================================= Booking =====================================
    public function booking(Request $request)
    {
        try {
            if (!isset($request['session_id'])) {
                $request['session_id'] = $this->headerSession;
            }
            $lead_guest = $request['lead_guest'];
            $request->validate([
                'lead_guest.title' => 'required|in:mr,mrs,miss',
                'lead_guest.first_name' => 'required|string|max:255',
                'lead_guest.last_name' => 'required|string|max:255',
                'lead_guest.email' => 'required|email|max:255',
                'lead_guest.mobile' => 'required',
                'lead_guest.nationality' => 'required|string',
                'session_id' => 'required',
            ], [
                'lead_guest.title.required' => __('api.title_required'),
                'lead_guest.title.in' => __('api.title_invalid'),
                'lead_guest.first_name.required' => __('api.first_name_required'),
                'lead_guest.first_name.string' => __('api.first_name_invalid'),
                'lead_guest.last_name.required' => __('api.last_name_required'),
                'lead_guest.last_name.string' => __('api.last_name_invalid'),
                'lead_guest.email.required' => __('api.email_required'),
                'lead_guest.email.email' => __('api.email_invalid'),
                'lead_guest.email_confirmation.required' => __('api.email_confirmation_required'),
                'lead_guest.email_confirmation.email' => __('api.email_confirmation_invalid'),
                'lead_guest.email_confirmation.same' => __('api.email_confirmation_mismatch'),
                'lead_guest.mobile.required' => __('api.mobile_required'),
                // 'lead_guest.mobile.string' => __('api.mobile_invalid'),
                'lead_guest.nationality.required' => __('api.nationality_required'),
                'lead_guest.nationality.string' => __('api.nationality_invalid'),
                'session_id.required' => __('api.session_required')
            ]);

            $user = createOrUpdateUser($lead_guest, $this->moduleKey);

            $sessionData = get_module_session($request->session_id);
            
            $input = [
                'UserId' => $user->id,
                'CurrencyCode' => 'USD',
                'LeadGuest' => $request['lead_guest'],
                'RoomsGuests' => $request['rooms'],
                'sessionData' => $sessionData,
                'agent' => $request->agent,
            ];

            // Call booking function
            $supplierKey = $sessionData['supplier'];
            $supplierClassObject = load_supplier($supplierKey, $this->moduleKey);
            $bookingResponse = $supplierClassObject->booking($input);
            $bookingResponse['data']['user'] = $user;
            return json_response([
                'status' => $bookingResponse['status'],
                'message' => $bookingResponse['message'],
                'data' => $bookingResponse['data']
            ], 200);
        } catch (Exception $e) {
            return json_response(
                [
                    'status' => false,
                    'message' => $e->getMessage()
                ],
                200
            );
        }
    }
    //========================================= Booking Details ============================
    public function booking_details($merchant_ref)
    {
        try {
            if (!$merchant_ref) {
                return new Exception('Merchant_Ref is required');
            }
            $booking_id = explode('.', $merchant_ref)[0];
            $Booking = HotelsBookingList::findOrFail($booking_id);
            return $Booking;
            $input = [
                'LangCode'              =>  $this->api->get_language_code(),
                'WowSession'            =>  $wow_session,
                'UserCredits'           => $this->api->current_user_credits,
                'UsersHelper'           => $this->api->users_helper,
                'UserId'                => $this->api->current_user_id,
                'MerchantRef'           =>  $merchant_ref
            ];

            $booking = $this->api->db
                ->where("booking_id", $booking_id)
                ->where("user_id", $this->api->get_merchant_ref($merchant_ref)['user_id'])
                ->get('hotels_booking_list')->row();

            $details = false;

            if ($booking) {
                // dj($booking);
                $input['Booking'] = $booking;
                // dj($booking);
                // $response = [];

                $details = $this->suppliers[$booking->supplier]->booking_details($input);
            }
            $response = [
                'status' => $details ? TRUE : FALSE,
                'message' => $details ? 'Done' : 'Error',
                'data' => $details ?? []
            ];

            $this->api->response([
                'status' => $details ? TRUE : FALSE,
                'message' => $details ? 'Done' : 'Error',
                'data' => $details['data'] ?? []
            ], API_Controller::HTTP_OK);
        } catch (Exception $e) {
            return json_response(
                [
                    'status' => false,
                    'message' => "Hotel Booking Details Error :" . $e->getMessage(),
                    'data' => []
                ],
                200
            );
        }
    }

    //======================================== Payment Details ======================================
    public function payment_details($merchant_ref)
    {
        try {
            if (!$merchant_ref) {
                return new Exception('Merchant_Ref is required');
            }
            $booking_id = explode('.', $merchant_ref)[0];
            $booking = HotelsBookingList::findOrFail($booking_id);
            $data['PaymentInfo'] = [
                // 'Ref' =>  $merchant_ref,
                'MerchantRef' => 'hotels.' . $merchant_ref,
                // 'Supplier' =>  $booking->supplier_id,
                'Price' => [
                    'Amount' => $booking->amount,
                    'Currency' => $booking->currency_code
                ],
            ];
            return json_response([
                'status' => true,
                'message' => 'Payment details successfully',
                'data' => $data
            ]);
        } catch (Exception $e) {
            return json_response([
                'status' => false,
                'message' => 'Payment details failed' . $e->getMessage(),
                'data' => []
            ]);
        }
    }
    //======================================== Pre Payment ======================================
    public function pre_payment($merchant_ref)
    {
        try {
            if (!$merchant_ref) {
                return new Exception('Merchant_Ref is required');
            }
            $booking_id = explode('.', $merchant_ref)[0];
            $booking = HotelsBookingList::findOrFail($booking_id);
            $available_for_pay = true; // You should define this variable based on your business logic
            $user_credits = $booking->company->balance->credit_amount; // Replace with your actual method

            $data['PaymentInfo'] = [
                // 'Ref' =>  $input['MerchantRef'],
                'Ref' =>  $merchant_ref,
                'Price' => [
                    // 'Amount' => $booking->amount,
                    // 'Amount' => payment_currency($booking->amount, 'USD', 'SAR'),
                    'Amount' => $booking->amount,
                    'Currency' => 'USD',
                ],
            ];

            $data['PaymentInfo']['Credit'] = [
                'AvailableForPay' => $available_for_pay,
                'UserCredits' => $user_credits,
                'CreditCurrency' => 'USD',
            ];

            $response = [
                'status' => $available_for_pay ? TRUE : FALSE,
                'message' => $available_for_pay ? 'Enough credits found' : 'You do not have enough credit',
                'data' => $data
            ];

            return json_response($response, 200);
        } catch (Exception $e) {
            return json_response([
                'status' => false,
                'message' => 'Payment details failed' . $e->getMessage(),
                'data' => []
            ]);
        }
    }

    //=============================================Invoice======================================================

    public function invoice($merchant_ref)
    {
        try {
            if (!$merchant_ref) {
                return new Exception('Merchant_Ref is required');
            }
            $booking_id = explode('.', $merchant_ref)[0];
            $booking = HotelsBookingList::findOrFail($booking_id);
            $user_credits = $booking->company->balance;
            // $available_for_pay = FALSE;
            $response = [];
            $supplierKey = $booking->supplier->key;
            $confirmation_number = $booking->confirmation_number;
            $user_net_amount = $user_credits->credit_amount - $booking->amount;
            // if ($user_net_amount >= $user_credits->credit_allowed_amount) {
            //     $available_for_pay = TRUE;
            // }
            $booking_details = $booking->booking_details;

            $input['BookingNeededDetails'] = $booking_details['booking_needed_details'];

            $supplierClassObject = load_supplier($supplierKey, $this->moduleKey);

            // if ($booking->confirmation_number == 'false' && $available_for_pay) {
            //     // no booking has been made yet
            //     // send voucher booking
            //     $input['VoucherBooking'] = true;
            //     $input['UsersHelper']->pay_from_credits($input['UserId'], $booking->amount);
            //     $response = $supplierClassObject->voucher_booking($booking_details['booking_needed_details'], $booking->booking_id);

            // } elseif ($booking->confirmation_number != 'false' && $available_for_pay) {
            // 
            // $input['VoucherBooking'] = true;
            // generate invoice
            // }
            pay_from_credits($booking->company, $booking->amount);
            $invoice_number = $supplierClassObject->generate_invoice($confirmation_number);
            $booking->update([
                'invoice_number' => $invoice_number,
                'booking_status' => 'Vouchered',
                'payment_status' => 'Paid',
                'voucher_status' => 'true'
            ]);
            $booking->save();

            $response = [
                'status' => true,
                'message' => 'success create invoice',
                'data' => $booking
            ];
            return json_response($response, 200);
        } catch (Exception $e) {
            return json_response([
                'status' => false,
                'message' => 'error create invoice number ',
            ], 200);
        }
    }
    //===================================================================================================
    public function country_list(Request $req)
    {
        $countries = Country::get();
        return json_response(['status' => true, 'message' => 'countries list successfully', 'data' => $countries]);
    }

    public function nationality_list(Request $req)
    {
        $nationalities = Nationality::get();
        return json_response(['status' => true, 'message' => 'nationalities list successfully', 'data' => $nationalities]);
    }
}
